function [EH, vk] = plane_wave(E, angle, phase, theta, phi)

% This function defines the the anonymous vectorial function in terms of
% the position of evaluation r  and the wavenumber k0 = 2*pi/lambda that 
% describes the electromagnetic field in every single point of the space. 

%%%%%% INPUT ARGUMENTS %%%%%%
%%% E:     electric  amplitude %%%
%%% angle: orientation of the electric amplitude %%%
%%% phase: phase difference between the second and first component %%%
%%% theta: angle of rotation with respect de x axis %%%
%%% phi:   angle of rotation with respect the z axis %%%


%%%%%% OUTPUT ARGUMENTS %%%%%%
%%% EH: 6D Vectorial (anonymous) funtion that describes the EM field as in 
%%%      terms of the impedance, the wavenumber and the 3D position %%%
%%% vk: unitary wavevector of the resulting plane wave.             %%%



%%%% Rotation of the origin wavevector: R(phi)*R(theta)*[0;0;0;1] %%%%
vk  = [sin(theta)*cos(phi); sin(theta)*sin(phi); cos(theta)];

%%%% Definition of the p and s component of the electric field %%%%
E01z = E*[cos(angle), sin(angle)*exp(1i*phase)];

%%%% Rotation of the electric field: R(phi)*R(theta)*E010z
E0   = [cos(phi)*cos(theta)*E01z(1) - sin(phi)*E01z(2)
       sin(phi)*cos(theta)*E01z(1) + cos(phi)*E01z(2)
                    -sin(theta)*E01z(1)];
                      
%%%% Creating resulting vectorial electromagnetic function %%%%
H0  = @(Z0)(1/Z0)*cross(vk,E0);
EH  = @(Z0,k0,r) vertcat(E0,H0(Z0))*exp(1i*k0*dot(vk,r));

               
end
                
